// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2012 Raritan Inc. All rights reserved.

package com.raritan.json_rpc;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import org.json.JSONException;
import org.json.JSONObject;

import com.raritan.idl.TypeInfo;

public abstract class ValueObjectCodec {

    /* methods to be implemented by value object codec classes */
    public abstract void encodeValObj(JSONObject json, ValueObject vo) throws JSONException;

    public abstract ValueObject decodeValObj(ValueObject vo, JSONObject json, Agent agent) throws JSONException;

    /*
     * static value object codec registry:
     * - key is stripped type name with all version numbers removed
     * - value is a map of fully versioned type infos to codecs
     */
    private static Map<String, TreeMap<TypeInfo, ValueObjectCodec>> codecMap =
            new HashMap<String, TreeMap<TypeInfo, ValueObjectCodec>>();

    public static void registerCodec(TypeInfo ti, ValueObjectCodec codec) {
        String strippedName = ti.getStrippedTypeName();
        TreeMap<TypeInfo, ValueObjectCodec> map = codecMap.get(strippedName);
        if (map == null) {
            // Using a TreeMap here to make sure the codecs are sorted by version,
            // which the logic in decodeAs() relies upon
            map = new TreeMap<TypeInfo, ValueObjectCodec>();
            codecMap.put(strippedName, map);
        }
        assert(!map.containsKey(ti));
        map.put(ti, codec);
    }

    private static TreeMap<TypeInfo, ValueObjectCodec> getCandidateCodecs(TypeInfo type) {
        if (codecMap.isEmpty()) {
            ValObjCodecRegistrar.act();
        }
        return codecMap.get(type.getStrippedTypeName());
    }

    /* package */ static ValueObjectCodec getCodec(TypeInfo type) {
        return getCandidateCodecs(type).get(type);
    }

    public static JSONObject encode(final ValueObject vo) throws JSONException {
        if (vo == null) return null;

        TypeInfo ti = vo.getTypeInfo();
        ValueObjectCodec voc = getCodec(ti);
        if (voc == null) {
            throw new IllegalArgumentException("No codec for " + ti);
        }

        JSONObject value = new JSONObject();
        voc.encodeValObj(value, vo);

        JSONObject json = new JSONObject();
        json.put("type", ti);
        json.put("value", value);
        return json;
    }

    public static ValueObject decodeAs(final JSONObject json, final Agent agent, final TypeInfo type) throws JSONException {
        if (json == null) {
            return null;
        }

        TypeInfo ti = new TypeInfo(json.getString("type"), null);
        TreeMap<TypeInfo, ValueObjectCodec> map = getCandidateCodecs(ti);
        TypeInfo bestTypeInfo = null;
        if (map != null) {
            for (TypeInfo entry : map.descendingKeySet()) {
                if (ti.isCompatible(entry)) {
                    bestTypeInfo = entry;
                    break;
                }
            }
        }
        if (bestTypeInfo == null) {
            throw new IllegalArgumentException("No codec for " + ti);
        }

        TypeInfo t = bestTypeInfo;
        while (t != null && !t.isCompatible(type)) {
            t = t.getBase();
        }
        if (t == null) {
            throw new IllegalArgumentException("Codec found for " + bestTypeInfo + ", but incompatible with " + type);
        }

        map = codecMap.get(type.getTypeName());
        ValueObjectCodec voc = map == null ? null : map.get(type);
        if (voc == null) {
            throw new IllegalArgumentException("No codec for desired type " + type);
        }

        JSONObject data = json.getJSONObject("value");
        ValueObject result = voc.decodeValObj(null, data, agent);
        if (result != null) {
            result.origData = data;
            result.origType = bestTypeInfo;
            result.agent = agent;
        }
        return result;
    }
}
