// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2012 Raritan Inc. All rights reserved.

package com.raritan.json_rpc;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.Base64;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;

import com.raritan.idl.AsyncRpcResponse;
import com.raritan.util.AsyncRequest;
import com.raritan.util.StringUtils;

public class Agent {

    private String remoteUrl = null;
    private String username = null;
    private String password = null;
    private String token = null;

    private int requestId = 0;

    private BulkRequestQueue defaultBulkRequestQueue = null;

    private static final String CHARSET = "UTF-8";
    private static final String HEADER_CLIENT_TYPE = "X-Client-Type";
    private static final String HEADER_AUTH = "Authorization";
    private static final String HEADER_SESSION_TOKEN = "X-SessionToken";
    private static final String CONTENT_TYPE = "Content-Type";

    private static final ThreadFactory sThreadFactory = new ThreadFactory() {
        private final AtomicInteger mCount = new AtomicInteger(1);

        public Thread newThread(Runnable r) {
            return new Thread(r, "Background RPC #" + mCount.getAndIncrement());
        }
    };

    private static final LinkedBlockingQueue<Runnable> sWorkQueue =
            new LinkedBlockingQueue<Runnable>();

    private final ThreadPoolExecutor mExecutor = new ThreadPoolExecutor(5, 5,
            5, TimeUnit.SECONDS, sWorkQueue, sThreadFactory);

    public interface RequestCallback {
        public void onFailure(Exception e);
        public void onSuccess(String response);
    }

    public Agent(String remoteUrl) {
        this.remoteUrl = remoteUrl;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public int getNextRequestId() {
        return requestId++;
    }

    public String getRemoteUrl() {
        return remoteUrl;
    }

    public String getUsername() {
        return username;
    }

    public String performRequest(String rid, String request) throws IOException, KeyManagementException, NoSuchAlgorithmException, RpcException {
        URL url;
        try {
            URI baseUri = new URI(remoteUrl);
            URI fullUri = new URI(baseUri.getScheme(), baseUri.getUserInfo(), baseUri.getHost(), baseUri.getPort(), rid, null, null);
            url = fullUri.toURL();
        } catch (URISyntaxException e) {
            throw new MalformedURLException(e.getMessage());
        }
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();

        if (connection instanceof HttpsURLConnection) {
            // accept self-signed certificates, do not verify host names
            HttpsURLConnection httpsConnection = (HttpsURLConnection) connection;
            httpsConnection.setSSLSocketFactory(new NaiveSSLSocketFactory());
            httpsConnection.setHostnameVerifier(new HostnameVerifier() {
                @Override
                public boolean verify(String arg0, SSLSession arg1) {
                    return true;
                }
            });
        }

        connection.setDoOutput(true);
        connection.setRequestProperty(HEADER_CLIENT_TYPE, "Java RPC");
        connection.setRequestProperty(CONTENT_TYPE, "application/json; charset=UTF-8");
        if (token != null) {
            connection.setRequestProperty(HEADER_SESSION_TOKEN, token);
        } else if (username != null && password != null) {
            String userPass = username + ":" + password;
            String basicAuth = "Basic " + Base64.getEncoder().encodeToString(userPass.getBytes());
            connection.setRequestProperty(HEADER_AUTH, basicAuth);
        }

        OutputStream out = null;
        InputStreamReader in = null;
        StringBuilder builder = new StringBuilder();

        try {
            byte[] requestBytes = request.getBytes();
            connection.setFixedLengthStreamingMode(requestBytes.length);
            out = connection.getOutputStream();
            out.write(requestBytes);

            int status = connection.getResponseCode();
            if (status != HttpURLConnection.HTTP_OK) {
                throw new RpcRequestException(status);
            }

            char buffer[] = new char[10000];
            in = new InputStreamReader(connection.getInputStream(), CHARSET);
            int read;
            do {
                read = in.read(buffer, 0, buffer.length);
                if (read > 0) {
                    builder.append(buffer, 0, read);
                }
            } while (read >= 0);
        } finally {
            if (out != null) {
                out.close();
            }
            if (in != null) {
                in.close();
            }
            connection.disconnect();
        }

        return builder.toString();
    }

    public AsyncRequest performRequest(final String rid, final String request, final AsyncRpcResponse<String> response) {
        final AsyncRequest req = new AsyncRequest(rid);

        scheduleInBackground(new Runnable() {
            @Override
            public void run() {
                try {
                    String response = performRequest(rid, request);
                    req.succeeded(response);
                } catch (Exception e) {
                    req.failed(e);
                }
            }
        });
        return req;
    }

    /* package */ void scheduleInBackground(Runnable r) {
        mExecutor.execute(r);
    }

    public void cleanup() {
        if (defaultBulkRequestQueue != null) {
            defaultBulkRequestQueue.shutdown();
            defaultBulkRequestQueue = null;
        }
        mExecutor.shutdownNow();
    }

    public BulkRequestQueue getDefaultBulkRequestQueue() {
        if (defaultBulkRequestQueue == null) {
            defaultBulkRequestQueue = new BulkRequestQueue(this);
        }
        return defaultBulkRequestQueue;
    }

    @Override
    public int hashCode() {
        return remoteUrl.hashCode() + username.hashCode();
    }

    @Override
    public boolean equals(Object other) {
        if (other instanceof Agent) {
            Agent otherAgent = (Agent) other;

            if (!StringUtils.stringsEqual(remoteUrl, otherAgent.remoteUrl)) {
                return false;
            }
            if (!StringUtils.stringsEqual(username, otherAgent.username)) {
                return false;
            }

            return true;
        }

        return false;
    }
}
