// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2012 Raritan Inc. All rights reserved.

package com.raritan.json_rpc;

import java.net.HttpURLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.json.JSONException;
import org.json.JSONObject;

import com.raritan.idl.AsyncRpcResponse;
import com.raritan.idl.bulkrpc.BulkRequest;
import com.raritan.idl.bulkrpc.BulkRequest.PerformRequestResult;
import com.raritan.idl.bulkrpc.Request;
import com.raritan.idl.bulkrpc.Response;
import com.raritan.json_rpc.bulkrpc.BulkRequest_Proxy;
import com.raritan.util.AsyncFailureListener;
import com.raritan.util.AsyncRequest;
import com.raritan.util.AsyncSuccessListener;

public class BulkRequestQueue implements Runnable {
    public interface JSONDecoder {
        JSONObject decodeAndValidate(String json) throws JSONException, RpcException;
    }

    private class SubRequest {
        String rid;
        JSONObject json;
        JSONDecoder decoder;
        AsyncRequest task;
    }

    private ScheduledThreadPoolExecutor mExecutor;
    private ScheduledFuture<?> mExecuteFuture;
    private final BulkRequest bulkRpc;

    private List<SubRequest> queued = new ArrayList<SubRequest>();

    static final int SEND_DELAY_MS = 250;

    public BulkRequestQueue(Agent agent) {
        mExecutor = new ScheduledThreadPoolExecutor(1);
        bulkRpc = new BulkRequest_Proxy(agent, "/bulk");
    }

    public AsyncRequest enqueueRequest(String rid, JSONObject object,
            JSONDecoder decoder, final AsyncRpcResponse<JSONObject> response) {
        AsyncRequest asyncRequest = new AsyncRequest(rid);

        asyncRequest.addSuccessListener(new AsyncSuccessListener() {
            @Override
            public void onSuccess(Object data) {
                response.onSuccess((JSONObject) data);
            }
        });

        asyncRequest.addFailureListener(new AsyncFailureListener() {
            @Override
            public void onFailure(Exception e) {
                response.onFailure(e);
            }
        });

        SubRequest req = new SubRequest();
        req.rid = rid;
        req.json = object;
        req.decoder = decoder;
        req.task = asyncRequest;

        synchronized (this) {
            queued.add(req);

            if (mExecuteFuture == null) {
                mExecuteFuture = mExecutor.schedule(this, SEND_DELAY_MS, TimeUnit.MILLISECONDS);
            }
        }

        return asyncRequest;
    }

    @Override
    public void run() {
        synchronized (this) {
            mExecuteFuture = null;
        }
        execute();
    }

    public void shutdown() {
        mExecutor.shutdownNow();
    }

    private void execute() {
        final List<SubRequest> sent;

        synchronized (this) {
            sent = queued;
            queued = new ArrayList<SubRequest>();
        }

        try {
            List<Request> requests = new ArrayList<Request>();
            for (SubRequest subreq : sent) {
                Request request = new Request();
                request.rid = subreq.rid;
                request.json = subreq.json.toString();
                requests.add(request);
            }

            PerformRequestResult result = bulkRpc.performRequest(requests);
            for (int i = 0; i < result.responses.size(); i++) {
                SubRequest subreq = sent.get(i);
                Response response = result.responses.get(i);
                if (response.statcode == HttpURLConnection.HTTP_OK) {
                    try {
                        JSONObject resp = subreq.decoder.decodeAndValidate(response.json);
                        subreq.task.succeeded(resp);
                    } catch (Exception e) {
                        subreq.task.failed(e);
                    }
                } else {
                    subreq.task.failed(new RpcRequestException(response.statcode));
                }
            }
        } catch (Exception e) {
            for (SubRequest subreq : sent) {
                subreq.task.failed(e);
            }
        }
    }
}
